///////////////////////////////////////////////////////////////////////////////
//
// smartnew.hpp - Replacement for standard new/delete operator, tuned for
//                allocation of many small, uni-sized objects.
//
//                Heavily based on code by Sasha Gontmakher and Ilan Horn,
//                Efficient Memory Allocation, Dr Dobbs Journal, January 1999.
//
// last updated: Mon 01/31/2000 11:26p
//
// +--------------------------------------------------------------------------+
// | Roland Weiss, WSI-CA         | eMail: weissr@informatik.uni-tuebingen.de |
// | University of Tuebingen      |      : weissr@uni-tuebingen.de            |
// | Sand 13, Room 218            |      : roland.weiss@uni-tuebingen.de      |
// | 72076 Tuebingen, Germany     | Tel  : +49 (0)7071 2974009                |
// |                              | Fax  : +49 (0)7071 295958                 |
// | WWW: http://www-ca.informatik.uni-tuebingen.de/people/weissr/weissr.html |
// +--------------------------------------------------------------------------+
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _WSI_SMARTNEW_HPP_
#define _WSI_SMARTNEW_HPP_

// Define _SMARTNEW_DEBUG to switch on debugging output messages.
#define _SMARTNEW_DEBUG

// It is only sensible to overload operator new[] with smartnew when your
// application usually uses small arrays, so it is turned off by default.
// Define _SMARTNEW_USE_ARRAY_NEW to turn it on. It's a good idea to use larger
// pages in this case.
//#define _SMARTNEW_USE_ARRAY_NEW

///////////////////////////////////////////////////////////////////////////////
//
// include files
//
///////////////////////////////////////////////////////////////////////////////
#include <new>
#include <string>

///////////////////////////////////////////////////////////////////////////////
namespace wsi {

// smartnew policy configuration
// object alignment granularity = 2 ^ granularity_bits
const size_t granularity_bits = 2;
const size_t granularity = 1 << granularity_bits;
// page size = 2 ^ page_size_bits
const size_t page_size_bits = 12;
const size_t page_size = 1 << page_size_bits;
// meta page size = 2 ^ metapage_size_bits
const size_t metapage_size_bits = 8;
const size_t metapage_size = 1 << metapage_size_bits;
// number of handled sizes = page_size / granularity
const size_t handled_sizes = page_size / granularity;

///////////////////////////////////////////////////////////////////////////////
//
// global helper methods
//
///////////////////////////////////////////////////////////////////////////////
// get next address >= val that is aligned to alignment
template <class T>
inline T align_up(T val, size_t alignment)
{
  return (val + alignment-1) & ~(alignment-1);
}
// get next address <= val that is aligned to alignment
template <class T>
inline T align_down(T ptr, size_t alignment)
{
  return ptr & ~(alignment-1);
}

// returns index into free_list for objects of size val with alignment
inline size_t free_list_index(size_t val, size_t alignment)
{
  return val / alignment;
}

// return global characteristics of smartnew in a string
std::string get_smartnew_policy();
// dump page list at given index: 0 <= index <= handled_sizes
void dump_page_list(size_t index);
// dump all active page lists
void dump_page_lists();


///////////////////////////////////////////////////////////////////////////////
//
// class Header
//
///////////////////////////////////////////////////////////////////////////////
class Header
{
public: // methods
  // clears current header
  void clear() { next = NULL; }
  // true if current header is set to NULL, otherwise false
  bool is_empty() const { return next == NULL; }
  // remove current header from list (write next header into current one)
  inline Header* deque();
  // insert new header at current position
  inline void enque(Header* obj);
  // return number of successors
  size_t count_successors();

private: // data members
  // points to next header in the list
  Header* next;
};

///////////////////////////////////////////////////////////////////////////////
//
// class Page
//
///////////////////////////////////////////////////////////////////////////////
class Page
{
public: // methods
  // doubly linked list methods

  // return pointer to next page on list
  Page* get_next() { return next; }
  // return pointer to previous page on list
  Page* get_prev() { return prev; }
  // return true if the page is empty, otherwise false
  bool is_empty() { return next == this; }
  // insert page into list after this page
  void link(Page* page);
  // remove this page from list
  void unlink();
  // check if page pointer are initialized, if not set them to this page
  void check_init();

  // standard object allocation handling

  // return size of objects stored in this page, 0 for big allocation pages
  size_t get_alloc_size() { return alloc_size; }
  // true if page is full, otherwise false
  bool is_page_full() const { return free_list.is_empty() && unallocated == NULL; }
  // true if page is empty, otherwise false
  bool is_page_empty() const { return alloc_count == 0; }
  // returns a pointer to a free object in the current page (must not be full)
  void* allocate();
  // put object back into page's free_list
  void free(void* obj);
  // initialize page for given size
  void designate(size_t size);

  // big allocation

  // mark page as big allocation site (alloc_size = 0)
  // place: pointer to raw memory for object and page header
  // returns pointer to aligned memory
  void* big_alloc(void* place);
  // returns start of mem allocated with malloc (in big_alloc)
  void* big_free() { return unallocated; }
  // true if this is the page header of a big allocation, otherwise false
  bool is_big_alloc() { return alloc_size == 0; }

  // return a string containing the page's properties
  std::string get_page_properties();

private: // data members
  // pointer to next page of this size
  Page* prev;
  // pointer to previous page of this size
  Page* next;

  // size of objects stored in this page, set to 0 for big allocation pages
  size_t alloc_size;
  // number of objects currently stored in this page
  size_t alloc_count;
  // list of free (deallocated) objects in this page
  Header free_list;
  // pointer to next unallocated object, set to NULL when page is full
  void* unallocated;

private: // methods
  void* get_first_obj() const {
    return reinterpret_cast<void*>(align_up(sizeof(Page) + reinterpret_cast<ptrdiff_t>(this), granularity));
  }
  void* get_last_obj() const {
    return reinterpret_cast<void*>(align_down(page_size - alloc_size + reinterpret_cast<ptrdiff_t>(this), granularity));
  }
  void* get_next_obj(void* obj) const {
    return reinterpret_cast<void*>(alloc_size + reinterpret_cast<size_t>(obj));
  }
};

// amount of free space per page
const size_t page_free_space = page_size - sizeof(Page);

///////////////////////////////////////////////////////////////////////////////
//
// class PagePool
//
///////////////////////////////////////////////////////////////////////////////
class PagePool
{
public: // methods
  // returns pointer to free page from pool if possible, otherwise NULL
  static Page* allocate();
  // put page back into free_list
  static void free(Page* page);

  // return number of available pages in pool
  static size_t available_pages() { return free_list.count_successors(); }

private: // data members
  // list of free pages belonging to any metapage
  static Header free_list;

private: // methods
  // allocate a metapage and enque its pages in the global freelist;
  // true if allocation succeeds, otherwise false
  static bool allocate_metapage();
};

} // end namespace wsi
///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////
//
// operators new & delete
//
///////////////////////////////////////////////////////////////////////////////
// new which throws bad_alloc on errors
void* operator new(size_t size) throw(std::bad_alloc);
// delete that matches new which throws bad_alloc
void operator delete(void* ptr) throw();

// new which returns NULL on errors
inline void* operator new(size_t size, const std::nothrow_t&) throw();
// delete that matches new which returns NULL
inline void operator delete(void* ptr, const std::nothrow_t&) throw();

#ifdef _SMARTNEW_USE_ARRAY_NEW
// array new
inline void* operator new[](size_t size) throw(std::bad_alloc);
inline void* operator new[](size_t size, const std::nothrow_t&) throw();
// array delete
inline void operator delete[](void* ptr) throw();
inline void operator delete[](void* ptr, const std::nothrow_t&) throw();
#endif // _SMARTNEW_USE_ARRAY_NEW


#endif // _WSI_SMARTNEW_HPP_

// Local Variables:
// mode: C++
// End:
