///////////////////////////////////////////////////////////////////////////////
//
// smartnew.cpp - Replacement for standard new/delete operator, tuned for
//                allocation of many small, uni-sized objects.
//
//                Heavily based on code by Sasha Gontmakher and Ilan Horn,
//                Efficient Memory Allocation, Dr Dobbs Journal, January 1999.
//
// last updated: Mon 01/31/2000 11:26p
//
// +--------------------------------------------------------------------------+
// | Roland Weiss, WSI-CA         | eMail: weissr@informatik.uni-tuebingen.de |
// | University of Tuebingen      |      : weissr@uni-tuebingen.de            |
// | Sand 13, Room 218            |      : roland.weiss@uni-tuebingen.de      |
// | 72076 Tuebingen, Germany     | Tel  : +49 (0)7071 2974009                |
// |                              | Fax  : +49 (0)7071 295958                 |
// | WWW: http://www-ca.informatik.uni-tuebingen.de/people/weissr/weissr.html |
// +--------------------------------------------------------------------------+
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
// include files
//
///////////////////////////////////////////////////////////////////////////////
#include "smartnew.hpp"

#include <cstdlib>
#include <cstdio>

#include <iostream>
#include <iomanip>
#include <strstream>

///////////////////////////////////////////////////////////////////////////////
namespace wsi {

///////////////////////////////////////////////////////////////////////////////
//
// create static (global) objects
//
///////////////////////////////////////////////////////////////////////////////
// list of free pages belonging to any metapage
Header PagePool::free_list;
// array holds a page list for every handled object size
// assumes that all member of page are initialized to zero (C++ std behavior)
Page page_lists[handled_sizes];

///////////////////////////////////////////////////////////////////////////////
//
// global functions
//
///////////////////////////////////////////////////////////////////////////////
using std::endl;

// return global characteristics of smartnew in a string
std::string get_smartnew_policy()
{
  std::ostrstream os;
  os << "-=# smartnew policy configuration #=-" << endl;
  os << "granularity             : " << granularity << endl;
  os << "page size               : " << page_size << endl;
  os << "free space per page     : " << page_free_space << endl;
  os << "pages per metapage      : " << metapage_size << endl;
  os << "number of handled sizes : " << handled_sizes << endl;
  return std::string(os.str(), os.pcount());
}

// dump page list at given index: 0 <= index <= handled_sizes
void dump_page_list(size_t index)
{
  if (index < handled_sizes)
  {
     // flag to indicate first loop iteration
    bool first_time = true;
    // counter for pages in list
    size_t page_count = 0;
    // remember start page to avoid infinite loop
    wsi::Page* start_pointer = page_lists[index].get_next();
    wsi::Page* page_pointer = start_pointer;
    // now iterate over all pages
    while ((page_pointer != NULL) &&
           ((page_pointer != start_pointer) || (first_time == true)))
    {
      first_time = false;
      // ignore empty dummy pages from startup
      if (page_pointer->get_alloc_size() > 0)
      {
        std::cout << page_pointer->get_page_properties() << endl;
        page_count++;
      } // if
      page_pointer = page_pointer->get_next();
    } // while
    if (page_count > 0)
      std::cout << "page count in list of size " << index * granularity
                << ": " << page_count << endl << endl;
  } // if
}

// dump all active page lists
void dump_page_lists()
{
  for (int i = 0; i < handled_sizes; i++)
    dump_page_list(i);
}

//////////////////////////////////////////////////////////////////////////////
//
// class Header
//
///////////////////////////////////////////////////////////////////////////////
// remove current header from list (write next header into current one)
inline Header* Header::deque() {
  Header* result = next;
  next = result->next;
  return result;
}

// insert new header at current position
inline void Header::enque(Header* obj) {
  obj->next = next;
  next = obj;
}

// return number of successors
size_t Header::count_successors()
{
  size_t count = 0;
  Header* tmp_pointer = next;
  while (tmp_pointer != NULL)
  {
    count++;
    tmp_pointer = tmp_pointer->next;
  }
  return count;
}

///////////////////////////////////////////////////////////////////////////////
//
// class Page
//
///////////////////////////////////////////////////////////////////////////////
// insert page into list after this page
void Page::link(Page* page)
{
  page->next = next;
  page->prev = this;
  next->prev = page;
  next = page;
}

// remove this page from list
void Page::unlink()
{
  next->prev = prev;
  prev->next = next;
}

// check if page pointers are initialized, if not set them to this page
void Page::check_init()
{
  if (prev == NULL)
    prev = next = this;
}

// returns a pointer to a free object in the current page (must not be full)
void* Page::allocate()
{
  // increase count of allocated objects
  alloc_count++;
  // try to get object from page's free_list
  if (!free_list.is_empty())
    return static_cast<void*>(free_list.deque());
  // get next object from raw page
  void* obj = unallocated;
  unallocated = get_next_obj(unallocated);
  // mark that no more objects are available on this page
  // they are now either in the page's free_list or used
  if (reinterpret_cast<size_t>(unallocated) > reinterpret_cast<size_t>(get_last_obj()))
    unallocated = 0;
  return obj;
}

// put object back into page's free_list
void Page::free(void* obj) {
  alloc_count--;
  free_list.enque(static_cast<Header*>(obj));
}

// initialize page for given size
void Page::designate(size_t size) {
  alloc_size = size;
  alloc_count = 0;
  free_list.clear();
  unallocated = get_first_obj();
}

// mark page as big allocation site (alloc_size = 0)
// place: pointer to raw memory for object and page header
// returns pointer to aligned memory
void* Page::big_alloc(void* place) {
  alloc_size = 0;
  unallocated = place;
  return get_first_obj();
}

// return a string containing the page's properties
std::string Page::get_page_properties()
{
  // use string stream as buffer and converter
  std::ostrstream os;
  if (alloc_size > 0)
  { // standard page
#ifdef _GNUC
    os.setf(std::ios_base::right);
#else
    os.setf(std::ios::right);
#endif
    os << "-- properties for page [" << this
       << "] of size " << std::setw(8) << alloc_size
       << " and capacity " << std::setw(8)
       << (reinterpret_cast<size_t>(get_last_obj()) - reinterpret_cast<size_t>(get_first_obj()))
          / alloc_size
       << " --" << endl;
    os << std::setw(8) << alloc_count << " object(s) allocated; "
       << std::setw(8) << free_list.count_successors()
       << " object(s) in free_list" << endl;
  }
  else
  { // big allocation page
    os << "== big allocation page [" << this
       << "] with memory at: " << unallocated << " ==" << endl;
  }
  return std::string(os.str(), os.pcount());
}

///////////////////////////////////////////////////////////////////////////////
//
// class PagePool
//
///////////////////////////////////////////////////////////////////////////////
// returns pointer to free page from pool if possible, otherwise NULL
Page* PagePool::allocate()
{
#ifdef _SMARTNEW_DEBUG
  ::printf(" [_SMARTNEW_DEBUG] providing page from metapage\n");
#endif // _SMARTNEW_DEBUG
  // if free_list is empty try to allocate a new metapage
  if (free_list.is_empty())
  {
    if (allocate_metapage() == false)
      return NULL;
  }
  Header* page = free_list.deque();
  return reinterpret_cast<Page*>(page);
}

// put page back into free_list
void PagePool::free(Page* page)
{
#ifdef _SMARTNEW_DEBUG
  ::printf(" [_SMARTNEW_DEBUG] put page back into metapage\n");
#endif // _SMARTNEW_DEBUG
  free_list.enque(reinterpret_cast<Header*>(page));
}

// allocate a metapage and enque its pages in the global freelist;
// true if allocation succeeds, otherwise false
bool PagePool::allocate_metapage()
{
  // allocate a metapage (size = number of pages * page_size)
  size_t num_pages = metapage_size;
  void *metapage = ::malloc(num_pages * page_size);
  if (metapage == NULL)
    return false;

  // align first page in metapage to page_size
  void *aligned_metapage =
    reinterpret_cast<void*>(align_up(reinterpret_cast<ptrdiff_t>(metapage), page_size));
  // one page is wasted if metapage was not already aligned
  if (metapage != aligned_metapage)
    num_pages -= 1;

  // put pages into freelist
  void *curr_page = aligned_metapage;
  for (size_t i = 0; i < num_pages; ++i)
  {
    free_list.enque(static_cast<Header*>(curr_page));
    curr_page = reinterpret_cast<void*>(reinterpret_cast<size_t>(curr_page) + page_size);
  }
#ifdef _SMARTNEW_DEBUG
  ::printf(" [_SMARTNEW_DEBUG] %d pages enqued in metapage\n", num_pages);
#endif // _SMARTNEW_DEBUG
  return true;
}

} // end namespace wsi
///////////////////////////////////////////////////////////////////////////////


///////////////////////////////////////////////////////////////////////////////
//
// operators new & delete
//
///////////////////////////////////////////////////////////////////////////////
using wsi::Page;

// new which throws bad_alloc on errors
void* operator new(size_t size) throw(std::bad_alloc)
{
  // requests for size 0 are delegated to size 1
  if (size == 0) size = 1;
  // adjust size according to granularity
  size = wsi::align_up(size, wsi::granularity);

  // handle allocation of objects greater than the free space per page
  if (size >= wsi::page_free_space)
  {
    // allocate enough mem to enable alignment
    void* place = ::malloc(size + wsi::page_size + sizeof(Page));
    // handle error case
    if (place == NULL)
    {
      // allocation was unsuccessful; get the current error-handling function
#ifdef _MSC_VER
      new_handler global_handler = set_new_handler(NULL);
      set_new_handler(global_handler);
#else // ANSI C++
      std::new_handler global_handler = std::set_new_handler(NULL);
      std::set_new_handler(global_handler);
#endif // _MSC_VER
      // call handler if set
      if (global_handler)
        (*global_handler)();
      else
        throw std::bad_alloc();
    }
    // align memory chunk to page_size
    Page* page = (Page*)wsi::align_up(reinterpret_cast<ptrdiff_t>(place),
                                      wsi::page_size);
    // init page
    return page->big_alloc(place);
  }

  // handle allocation of standard objects (small)
  Page& header = wsi::page_lists[wsi::free_list_index(size, wsi::granularity)];
  // check page for initialization (and do it if necessary)
  header.check_init();

  // allocate page when empty
  if (header.is_empty())
  {
#ifdef _SMARTNEW_DEBUG
    ::printf(" [_SMARTNEW_DEBUG] allocating page of size %d\n", size);
#endif // _SMARTNEW_DEBUG
    // get new page from PagePool
    Page* free_page = wsi::PagePool::allocate();
    header.link(free_page);
    // initialize new page
    free_page->designate(size);
  }

  // get pointer to free page
  Page* free_page = header.get_next();
  // get pointer to object in page
  void* obj = free_page->allocate();
  // if page is now full, remove it from this size's free_list
  if (free_page->is_page_full())
    free_page->unlink();
  return obj;
}

// delete that matches new which throws bad_alloc
void operator delete(void* ptr) throw()
{
  // handle NULL pointer deallocation
  if (ptr == NULL) return;
  // get pointer to page start (header with information)
  Page* page = reinterpret_cast<Page*>
    (wsi::align_down(reinterpret_cast<ptrdiff_t>(ptr), wsi::page_size));

  // handle big deallocation
  if (page->is_big_alloc())
  {
    void* place = page->big_free();
    ::free(place);
    return;
  }

  // handle standard deallocation
  if (page->is_page_full())
  {
    // page will go back into free_list
    page->free(ptr);
    // put empty page back into PagePool
    if (page->is_page_empty())
      wsi::PagePool::free(page);
    else
    {
      // put page back into free_list of its size
      Page& header = wsi::page_lists[wsi::free_list_index(page->get_alloc_size(), wsi::granularity)];
      header.link(page);
    }
    return;
  }
  else
  {
    page->free(ptr);
    if (page->is_page_empty()) {
      // remove page from free_list of its size
      page->unlink();
      // put empty page back into PagePool
      wsi::PagePool::free(page);
    }
  }
}

// new which returns NULL on errors
inline void* operator new(size_t size, const std::nothrow_t&) throw()
{
  // try the operator new which throws bad_alloc
  try
  {
    return operator new(size);
  }
  // if it fails, return NULL
  catch (...)
  {
    return NULL;
  }
}

// delete that matches new which returns NULL
inline void operator delete(void* ptr, const std::nothrow_t&) throw()
{
  operator delete(ptr);
}

#ifdef _SMARTNEW_USE_ARRAY_NEW
// array new
inline void* operator new[](size_t size) throw(std::bad_alloc)
{
  return operator new(size);
}
inline void* operator new[](size_t size, const std::nothrow_t&) throw()
{
  // try the operator new which throws bad_alloc
  try
  {
    return operator new(size);
  }
  // if it fails, return NULL
  catch (...)
  {
    return NULL;
  }
}

// array delete
inline void operator delete[](void* ptr) throw()
{
  operator delete(ptr);
}

inline void operator delete[](void* ptr, const std::nothrow_t&) throw()
{
  operator delete(ptr);
}
#endif // _SMARTNEW_USE_ARRAY_NEW

// Local Variables:
// mode: C++
// compile-command: "g++ -c smartnew.cpp"
// End:
