////////////////////////////////////////////////////////////////////////////////
//                                                                            //
// File       : nseq.hpp                                                      //
// Author     : Roland Weiss (weissr@informatik.uni-tuebingen.de)             //
// Copyright  : Roland Weiss (weissr@informatik.uni-tuebingen.de)             //
// Time-Stamp : <Tue Jul 11 15:01:55 MEST 2000>                               //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#ifndef WSI_NSEQ_HPP
#define WSI_NSEQ_HPP

// standard library
#include <iostream>
#include <stack>

namespace wsi {

//------------------------------------------------------------------------------
// base_elem is the abstract base class for all elements that will be stored in
// nested sequences. It provides methods for polymorphic behavior.
//------------------------------------------------------------------------------
class base_elem
{
public:
  // Construct a new element.
  //  returns: pointer to new element
  virtual base_elem* create() const = 0;
  // Copy the element.
  //  shallow: if false, a deep copy will be made, otherwise a shallow copy
  //    [Note: this parameter has no effect for atomic elements]
  //  returns: pointer to element's copy
  virtual base_elem* clone(bool shallow = false) const = 0;
  // Virtual destructor.
  virtual ~base_elem() {};

  // Test the element's atomic attribute (atomic or sequence element).
  //  returns: true if the element is an atom, otherwise false
  virtual bool is_atom() const = 0;

  // Writes the element's representation into an output stream.
  //  os: output stream which should be modified
  //  returns: modified output stream
  virtual std::ostream& print(std::ostream& os) const { return os; };
};

// Overloaded output operator for base_elem for integration with the std I/O
// streams of C++. Relies on the polymophic behavior base_elem's print method.
//  os: output stream to be modified
//  el: element to write into the output stream
//  returns: modified output stream
std::ostream& operator<<(std::ostream& os, const base_elem& el)
{
  return el.print(os);
}

//------------------------------------------------------------------------------
// elem is a template class that can be used to wrap objects of any type for
// usage within nested sequences. Inherits from base_elem.
//  T: type to be wrapped
//------------------------------------------------------------------------------
template <class T>
class elem : public base_elem
{
public:
  // Default constructor.
  elem<T>() : val(T()) {}
  // Copy constructor.
  elem<T>(const elem<T>& e) : val(T(e.val)) {}
  // Copy constructor with value of wrapped type as argument.
  elem<T>(const T& v) : val(T(v)) {}

  // Assignment operator.
  elem<T>& operator=(const elem<T>& e) { val = T(e.val); return *this; }
  // Assignment operator with value of wrapped type as argument.
  elem<T>& operator=(const T& v) { val = T(v); return *this; }

  // Get the element's current value.
  //  returns: element's value
  const T& getValue() const { return val; }
  // Dereference operator, used to get the element's current value.
  //  returns: element's value
  const T& operator*() const { return val; }

  // Conversion operator. Implicit conversions are reasonable for a wrapper class.
  operator T() const { return val; }

  // Override [base_elem] virtual functions for creation and cloneing of elements.
  // Construct a new element.
  //  returns: pointer to new element
  elem<T>* create() const { return new elem<T>(); }
  // Copy the element.
  //  shallow: [Note: this parameter has no effect for atomic elements]
  //  returns: pointer to element's copy
  elem<T>* clone(bool shallow = false) const { return new elem<T>(*this); }

  // Override [base_elem] the virtual function for the element's atomicity test.
  //  returns: true
  bool is_atom() const { return true; }

  // Override [base_elem] virtual print method for elements.
  //  os: output stream which should be modified
  //  returns: modified output stream
  std::ostream& print(std::ostream& os) const { return os << val; }

private:
  // Holds value of wrapped object.
  T val;
};

//------------------------------------------------------------------------------
// The traits class nseq_output_traits defines the appearance of a streamed nseq.
//  open_nlevel: string for opening a nesting level
//  close_nlevel: string for closing a nesting level
//  elem_seperator: string for seperating elements
//------------------------------------------------------------------------------
// The default behavior is to use brackets for nesting and space as separator.
struct nseq_output_traits
{
  static const char open_nlevel;
  static const char close_nlevel;
  static const char elem_seperator;
};
const char nseq_output_traits::open_nlevel = '[';
const char nseq_output_traits::close_nlevel = ']';
const char nseq_output_traits::elem_seperator = ' ';

//------------------------------------------------------------------------------
// nseq is the template class for nested sequences (tree). It uses a STL
// conforming sequence container as internal representation.
// Inherits from any given STL conforming sequence container and base_elem.
//  C<T, Allocator>: STL sequence container used for internal representation
//  output_traits: traits for streamed appearance of nseq
//------------------------------------------------------------------------------

// runs with ANSI C++ compilers: edg 2.43, Visual Age C++ 4.0, Borland C++ 5.5
template < template <typename T, typename A> class cont_t,
           typename output_traits = nseq_output_traits,
           template <typename T> class alloc_t = std::allocator >
class nseq : public cont_t< base_elem*, alloc_t<base_elem*> >,
             public base_elem
{
private:
  typedef cont_t< base_elem*, std::allocator<base_elem*> > container_type;

public:
  // export iterator names
  typedef typename container_type::iterator iterator;
  typedef typename container_type::const_iterator const_iterator;

  // Override [base_elem] virtual functions for creation and cloneing of nseqs.
  // Construct a nseq.
  //  returns: pointer to new element
  nseq* create() const { return new nseq(); }
  // Copy the nseq.
  //  shallow: if false, a deep copy will be made, which calls create recursively,
  //    otherwise a shallow copy is made (only top-level pointers are copied)
  //  returns: pointer to nseq's copy
  nseq* clone(bool shallow = false) const
  {
    nseq *tmp = new nseq;
    for (const_iterator it = begin(); it != end(); ++it)
    {
      if (shallow == true) // shallow copy (just pointers)
        tmp->push_back(*it);
      else // deep copy (clone elements)
        tmp->push_back((*it)->clone());
    }
    return tmp;
  }

  // Override [base_elem] the virtual function for the element's atomicity test.
  //  returns: false
  bool is_atom() const { return false; }

  // Override [base_elem] virtual print method for nseqs. Uses output_traits.
  //  os: output stream which should be modified
  //  returns: modified output stream
  std::ostream& print(std::ostream& os) const
  {
    os << output_traits::open_nlevel;
    for (const_iterator it = begin(); it != end();)
    {
      os << **it;
      if (++it != end()) os << output_traits::elem_seperator;
    }
    os << output_traits::close_nlevel;
    return os;
  }
};

//------------------------------------------------------------------------------
// A flat_iterator traverses nested sequences element by element. It is either
// positioned at a valid element (which may be an atom or a nested sequence) or
// at the top-level end() iterator.
//  NSEQ: nested sequence container type
//  I: iterator type of internal sequence container
//------------------------------------------------------------------------------
template <typename NSEQ, typename I = typename NSEQ::iterator>
class flat_iterator : public iterator_traits<I>
{
private:
  // Iterator pointing to the current position in the nseq.
  I cpos;
  // Iterator pointing to the end of the current sub-level sequence.
  I cend;
  // Stack holds the iterators cpos and cend of the parent sequences.
  stack< pair<I, I> > iter_stack;
public:
  // Export type names.
	typedef forward_iterator_tag iterator_category;
  typedef base_elem*& reference; // not in Plauger STL iterator_traits
  typedef base_elem** pointer; // not in Plauger STL iterator_traits

  // Default constructor.
  flat_iterator() {}
  // Copy constructor.
  flat_iterator(const flat_iterator& it) : cpos(it.cpos) {}
  // Copy constructor with value of wrapped type as argument.
  flat_iterator(const I& it) : cpos(it) {}

  // Dereference operator.
  //  returns: reference to element (which is a pointer) at the current position
  reference operator*() { return *cpos; }
  // Access operator.
  //  returns: pointer to element (which is a pointer) at the current position
  pointer operator->() const { return &(operator*()); }

  // Comparison operators for flat_iterators.
  //  it: flat_iterator to compare with
  //  returns: true if comparision is true, wotherise false
  bool operator==(const flat_iterator& it) const { return cpos == it.cpos; }
  bool operator!=(const flat_iterator& it) const { return cpos != it.cpos; }
  // Comparison operators for flat_iterators with sequence iterators.
  //  it: sequence iterator to compare with
  //  returns: true if comparision is true, wotherise false
  bool operator==(const I& it) const { return cpos == it; }
  bool operator!=(const I& it) const { return cpos != it; }

  // Pre-increment operator.
  //  returns: reference of iterator pointing to next position
  //
  // [Implementation note: Because the increment operations have to check the
  // element's atomicity, it is extremely important to make sure that a returned
  // iterator is pointing to a valid position. Therefore, check all cases where
  // the internal iterator was increased.
  flat_iterator& operator++()
  {
    if ((*cpos)->is_atom()) ++cpos; // either handle atomic elements
    else // or deal with nested sequences
    {
      // cast element to nseq and increase internal iterator
      if (NSEQ *ns = dynamic_cast<NSEQ*>(*(cpos++)))
      {
        // get begin and end iterators of nested sequence
        I ns_begin = ns->begin();
        I ns_end = ns->end();
        // recurse into non-empty nseqs
        if (ns_begin != ns_end)
        {
          iter_stack.push(make_pair(cpos, cend));
          cpos = ns_begin;
          cend = ns_end;
        }
      }
    }
    // finally avoid returning an iterator positioned at the end of a sub-nseq
    if (!iter_stack.empty() && (cpos == cend))
    {
      do
      {
        cpos = iter_stack.top().first;
        cend = iter_stack.top().second;
        iter_stack.pop();
      } while (!iter_stack.empty() && (cpos == cend));
    }
    return *this;
  }
  // Post-increment operator.
  //  returns: iterator pointing to current position
  flat_iterator operator++(int)
  {
    // make copy and use pre-increment operator
    flat_iterator tmp = *this;
    operator++();
    return tmp;
  }
};

} // namespace wsi

#endif // WSI_NSEQ_HPP

// Local Variables:
// mode: C++
// End:
